<?php
/*
 *  JIHUDUMIE HAPA - WiFi Self Service Page
 *  Copyright (C) 2018 Laksamadi Guko.
 *
 *  Save as: /xampp/htdocs/mikhmonB/jihudumie_hapa.php
 */

session_start();
// hide all error
error_reporting(0);

ob_start("ob_gzhandler");

// Get the session from URL (exactly like admin.php does)
$session = $_GET['session'];

// If no session provided, use default
if (empty($session)) {
    $session = 'session1';
}

// Load config files in the EXACT same order as admin.php
include('./include/config.php');
include('./include/readcfg.php');
include_once('./lib/routeros_api.class.php');
include_once('./lib/formatbytesbites.php');

// Connection status variables
$connection_status = 'Not Connected';
$connection_color = 'red';
$API = null;

// Check if we have the necessary variables and class
if (class_exists('RouterosAPI') && isset($iphost) && isset($userhost) && isset($passwdhost) && function_exists('decrypt')) {
    // Create API connection following admin.php pattern
    ini_set("max_execution_time", 5);
    $API = new RouterosAPI();
    $API->debug = false;
    
    // Try to connect to router using config from readcfg.php
    if ($API->connect($iphost, $userhost, decrypt($passwdhost))){
        $connection_status = 'Connected';
        $connection_color = 'green';
    } else {
        $connection_status = 'Not Connected - Check router IP/credentials';
    }
} else {
    // Debug info
    if (!class_exists('RouterosAPI')) {
        $connection_status = 'RouterOS API class not found';
    } elseif (!isset($iphost)) {
        $connection_status = 'Config not loaded - check session parameter';
    } elseif (!function_exists('decrypt')) {
        $connection_status = 'Decrypt function not found';
    }
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['action'];
    
    if ($action === 'search_voucher') {
        $voucher = trim($_POST['voucher'] ?? '');
        
        // Require at least 7 characters
        if (strlen($voucher) < 7) {
            echo json_encode(['success' => false, 'message' => 'Voucher lazima iwe na angalau herufi 7']);
            exit;
        }
        
        if (!isset($API) || $connection_status !== 'Connected') {
            echo json_encode(['success' => false, 'message' => 'Kuna tatizo la mtandao wa router. Router haipatikani.']);
            exit;
        }
        
        try {
            // Search using the same pattern as your users page
            $getuser = $API->comm("/ip/hotspot/user/print", array("?name" => $voucher));
            
            if (empty($getuser)) {
                $getuser = $API->comm("/ip/hotspot/user/print", array("?comment" => $voucher));
            }
            
            if (empty($getuser)) {
                // Search all users and check if voucher is in name or comment
                $all_users = $API->comm("/ip/hotspot/user/print");
                foreach ($all_users as $user) {
                    $uname = $user['name'] ?? '';
                    $ucomment = $user['comment'] ?? '';
                    
                    if (stripos($uname, $voucher) !== false || stripos($ucomment, $voucher) !== false) {
                        $getuser = array($user);
                        break;
                    }
                }
            }
            
            if (!empty($getuser)) {
                $user = $getuser[0];
                echo json_encode([
                    'success' => true,
                    'user' => array(
                        'id' => $user['.id'],
                        'name' => $user['name'],
                        'password' => $user['password'],
                        'profile' => $user['profile'],
                        'server' => $user['server'] ?? 'all',
                        'uptime' => $user['uptime'] ?? '0',
                        'limit_uptime' => $user['limit-uptime'] ?? '',
                        'limit_bytes' => $user['limit-bytes-total'] ?? '',
                        'comment' => $user['comment'] ?? '',
                        'disabled' => $user['disabled'] ?? 'no'
                    )
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Voucher "' . $voucher . '" haijapatikana. Hakikisha umeweka jina sahihi.']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Kuna tatizo la mtandao. Jaribu tena.']);
        }
        exit;
    }
    
    if ($action === 'repair_step') {
        if (!isset($API) || $connection_status !== 'Connected') {
            echo json_encode(['success' => false, 'message' => 'Kuna tatizo la mtandao wa router.']);
            exit;
        }
        
        $step = $_POST['step'];
        $user_data = json_decode($_POST['user_data'], true);
        
        try {
            if ($step === 'remove_user') {
                // Step 1: Remove old user
                // First check if user exists
                $check_before = $API->comm("/ip/hotspot/user/print", array("?name" => $user_data['name']));
                
                if (!empty($check_before)) {
                    // User exists, try to remove it
                    $remove_result = $API->comm("/ip/hotspot/user/remove", array(".id" => $user_data['id']));
                    
                    // Wait and verify removal (try up to 3 times)
                    $removed = false;
                    for ($i = 0; $i < 3; $i++) {
                        usleep(500000); // 0.5 seconds
                        $check_after = $API->comm("/ip/hotspot/user/print", array("?name" => $user_data['name']));
                        if (empty($check_after)) {
                            $removed = true;
                            break;
                        }
                    }
                    
                    if ($removed) {
                        echo json_encode(['success' => true, 'step_completed' => 'remove_user']);
                    } else {
                        echo json_encode(['success' => false, 'message' => 'Voucher bado ipo kwenye mikrotik. Jaribu tena.']);
                    }
                } else {
                    // User already doesn't exist, consider it success
                    echo json_encode(['success' => true, 'step_completed' => 'remove_user']);
                }
                
            } elseif ($step === 'remove_cookies') {
                // Step 2: Remove cookies
                // Find all cookies for this user
                $getcookies = $API->comm("/ip/hotspot/cookie/print", array("?user" => $user_data['name']));
                
                if ($getcookies && is_array($getcookies) && count($getcookies) > 0) {
                    // Cookies exist, remove them
                    foreach ($getcookies as $cookie) {
                        if (isset($cookie['.id'])) {
                            $API->comm("/ip/hotspot/cookie/remove", array(".id" => $cookie['.id']));
                        }
                    }
                    
                    // Wait and verify removal (try up to 3 times)
                    $cookies_removed = false;
                    for ($i = 0; $i < 3; $i++) {
                        usleep(500000); // 0.5 seconds
                        $check_cookies = $API->comm("/ip/hotspot/cookie/print", array("?user" => $user_data['name']));
                        if (empty($check_cookies)) {
                            $cookies_removed = true;
                            break;
                        }
                    }
                    
                    if ($cookies_removed) {
                        echo json_encode(['success' => true, 'step_completed' => 'remove_cookies']);
                    } else {
                        echo json_encode(['success' => false, 'message' => 'Cookies bado zipo. Jaribu tena.']);
                    }
                } else {
                    // No cookies found, consider it success
                    echo json_encode(['success' => true, 'step_completed' => 'remove_cookies']);
                }
                
            } elseif ($step === 'add_user') {
                // Step 3: Add user
                // First check if voucher already exists (shouldn't, but double-check)
                $check_exists = $API->comm("/ip/hotspot/user/print", array("?name" => $user_data['name']));
                
                if (!empty($check_exists)) {
                    // User exists, remove it first
                    $API->comm("/ip/hotspot/user/remove", array(".id" => $check_exists[0]['.id']));
                    usleep(500000); // Wait 0.5 seconds
                }
                
                // Now add the user
                $timelimit = ($user_data['limit_uptime'] == '' || $user_data['limit_uptime'] == '0') ? '0' : $user_data['limit_uptime'];
                $datalimit = ($user_data['limit_bytes'] == '' || $user_data['limit_bytes'] == '0') ? '0' : $user_data['limit_bytes'];
                
                $add_params = array(
                    "server" => $user_data['server'],
                    "name" => $user_data['name'],
                    "password" => $user_data['password'],
                    "profile" => $user_data['profile'],
                    "disabled" => "no"
                );
                
                if ($timelimit != '0') {
                    $add_params["limit-uptime"] = $timelimit;
                }
                if ($datalimit != '0') {
                    $add_params["limit-bytes-total"] = $datalimit;
                }
                if (!empty($user_data['comment'])) {
                    $add_params["comment"] = $user_data['comment'];
                }
                
                $add_result = $API->comm("/ip/hotspot/user/add", $add_params);
                
                // Verify the user was added (try up to 5 times)
                $user_added = false;
                for ($i = 0; $i < 5; $i++) {
                    usleep(500000); // 0.5 seconds
                    $verify_user = $API->comm("/ip/hotspot/user/print", array("?name" => $user_data['name']));
                    if ($verify_user && count($verify_user) > 0) {
                        $user_added = true;
                        break;
                    }
                }
                
                if (!$user_added) {
                    echo json_encode(['success' => false, 'message' => 'Voucher haijaongezwa. Jaribu tena.']);
                    exit;
                }
                
                // Finally, verify no cookies exist for this user
                usleep(500000);
                $check_final_cookies = $API->comm("/ip/hotspot/cookie/print", array("?user" => $user_data['name']));
                
                if (empty($check_final_cookies)) {
                    // All good!
                    echo json_encode([
                        'success' => true, 
                        'step_completed' => 'add_user',
                        'voucher' => $user_data['name'],
                        'password' => $user_data['password']
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Voucher imeongezwa lakini cookies bado zipo. Jaribu tena.']);
                }
            }
            
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Kuna tatizo la mtandao. Jaribu tena. (' . $e->getMessage() . ')']);
        }
        exit;
    }
    
    if ($action === 'log_failure') {
        // Log failure to a file for admin review
        $step = $_POST['step'];
        $user_data = json_decode($_POST['user_data'], true);
        $retry_count = $_POST['retry_count'];
        
        $log_entry = date('Y-m-d H:i:s') . " | Step: $step | Voucher: " . $user_data['name'] . " | Retries: $retry_count\n";
        
        // Log to file
        $log_file = __DIR__ . '/logs/jihudumie_failures.log';
        $log_dir = dirname($log_file);
        
        if (!file_exists($log_dir)) {
            @mkdir($log_dir, 0755, true);
        }
        
        @file_put_contents($log_file, $log_entry, FILE_APPEND);
        
        echo json_encode(['success' => true]);
        exit;
    }
    
    if ($action === 'create_temp_voucher') {
        if (!isset($API) || $connection_status !== 'Connected') {
            echo json_encode(['success' => false, 'message' => 'Router connection failed']);
            exit;
        }
        
        $data = json_decode($_POST['data'], true);
        $phone = $data['phone'];
        $original_voucher = $data['original_voucher'];
        
        try {
            // Generate unique temp voucher name
            $temp_voucher = 'temp_' . time();
            
            // Create the temporary voucher with profile "masaa3" and 3-hour limit
            $add_result = $API->comm("/ip/hotspot/user/add", array(
                "server" => "all",
                "name" => $temp_voucher,
                "password" => $temp_voucher,
                "profile" => "masaa3",
                "disabled" => "no",
                "limit-uptime" => "3h",
                "comment" => "TEMP-" . $phone . "-FOR-" . $original_voucher
            ));
            
            // Verify it was created
            usleep(500000);
            $verify = $API->comm("/ip/hotspot/user/print", array("?name" => $temp_voucher));
            
            if ($verify && count($verify) > 0) {
                // Log temp voucher creation
                $log_entry = date('Y-m-d H:i:s') . " | Temp Voucher Created: $temp_voucher | Phone: $phone | Original: $original_voucher\n";
                $log_file = __DIR__ . '/logs/temp_vouchers.log';
                @file_put_contents($log_file, $log_entry, FILE_APPEND);
                
                echo json_encode([
                    'success' => true,
                    'voucher' => $temp_voucher
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create temp voucher']);
            }
            
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }
}
?>

<!DOCTYPE html>
<html lang="sw">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Jihudumie Hapa - WiFi Self Service</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #ff6b6b, #feca57);
            color: white;
            padding: 30px 20px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .connection-status {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 10px;
            font-size: 1em;
        }
        
        .status-dot {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
            display: inline-block;
        }
        
        .status-dot.green {
            background-color: #28a745;
            box-shadow: 0 0 5px rgba(40, 167, 69, 0.5);
        }
        
        .status-dot.red {
            background-color: #dc3545;
            box-shadow: 0 0 5px rgba(220, 53, 69, 0.5);
        }
        
        .status-text {
            font-weight: 500;
        }
        
        .content {
            padding: 30px 20px;
        }
        
        .explanation {
            background: #f8f9fa;
            border-left: 5px solid #17a2b8;
            padding: 20px;
            margin-bottom: 30px;
            border-radius: 5px;
        }
        
        .explanation h2 {
            color: #495057;
            margin-bottom: 15px;
            font-size: 1.2em;
        }
        
        .explanation p {
            color: #6c757d;
            line-height: 1.6;
        }
        
        .search-section {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .search-label {
            font-size: 1.3em;
            color: #495057;
            margin-bottom: 15px;
            display: block;
            font-weight: 600;
        }
        
        .search-group {
            display: flex;
            gap: 10px;
            justify-content: center;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .search-input {
            width: 100%;
            max-width: 300px;
            padding: 15px 20px;
            font-size: 1.3em;
            text-align: center;
            border: 3px solid #dee2e6;
            border-radius: 25px;
            outline: none;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0,123,255,0.25);
        }
        
        .search-btn {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            border: none;
            padding: 15px 30px;
            font-size: 1.1em;
            border-radius: 25px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .search-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,123,255,0.3);
        }
        
        .progress-section {
            display: none;
            text-align: center;
            padding: 30px;
        }
        
        .progress-steps {
            margin: 20px 0;
        }
        
        .progress-step {
            background: #f8f9fa;
            padding: 15px;
            margin: 10px 0;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .progress-step.active {
            background: #fff3cd;
            border-left: 5px solid #ffc107;
        }
        
        .progress-step.completed {
            background: #d4edda;
            border-left: 5px solid #28a745;
        }
        
        .progress-step.failed {
            background: #f8d7da;
            border-left: 5px solid #dc3545;
        }
        
        .step-icon {
            font-size: 1.5em;
            margin-right: 10px;
        }
        
        .spinner {
            width: 30px;
            height: 30px;
            border: 4px solid #f3f3f3;
            border-top: 4px solid #007bff;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .result {
            display: none;
            text-align: center;
        }
        
        .user-info {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .user-info h3 {
            color: #1976d2;
            margin-bottom: 15px;
        }
        
        .user-details {
            text-align: left;
            display: inline-block;
        }
        
        .user-details p {
            margin: 8px 0;
            color: #424242;
        }
        
        .user-details strong {
            color: #1976d2;
        }
        
        .repair-btn {
            background: linear-gradient(135deg, #ff6b6b, #feca57);
            color: white;
            border: none;
            padding: 15px 30px;
            font-size: 1.2em;
            border-radius: 25px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
            text-transform: uppercase;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .repair-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 7px 20px rgba(0,0,0,0.3);
        }
        
        .retry-btn {
            background: linear-gradient(135deg, #ffc107, #ff9800);
            color: white;
            border: none;
            padding: 12px 25px;
            font-size: 1em;
            border-radius: 20px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
            margin-top: 15px;
        }
        
        .retry-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(255,193,7,0.3);
        }
        
        .success {
            display: none;
            text-align: center;
            padding: 30px;
        }
        
        .success-icon {
            font-size: 4em;
            color: #28a745;
            margin-bottom: 20px;
        }
        
        .success h2 {
            color: #28a745;
            margin-bottom: 20px;
        }
        
        .voucher-display {
            background: #d4edda;
            border: 2px solid #28a745;
            border-radius: 10px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .voucher-code {
            font-size: 2em;
            font-weight: bold;
            color: #155724;
            margin: 10px 0;
            letter-spacing: 2px;
        }
        
        .copy-redirect-btn {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
            padding: 15px 30px;
            font-size: 1.1em;
            border-radius: 25px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
            margin: 10px;
        }
        
        .copy-redirect-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 7px 20px rgba(0,0,0,0.3);
        }
        
        .error {
            color: #dc3545;
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            border-radius: 5px;
            padding: 15px;
            margin: 15px 0;
            text-align: center;
        }
        
        @media (max-width: 768px) {
            .container {
                margin: 10px;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .content {
                padding: 20px 15px;
            }
            
            .search-group {
                flex-direction: column;
            }
            
            .search-input {
                max-width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔧 JIHUDUMIE HAPA 🔧</h1>
            <div class="connection-status">
                <span class="status-dot <?= $connection_color ?>"></span>
                <span class="status-text">Router: <?= $connection_status ?></span>
            </div>
        </div>
        
        <div class="content">
            <div class="explanation">
                <h2>📶 Kama Voucher Yako Inasumbua?</h2>
                <p>Kama voucher yako haikubali, imekutoa kabla ya muda, au umehama jina la WiFi ulilokua unaitumia kwenda nyingine, weka voucher yako hapo chini ili tuirekebishe.</p>
            </div>
            
            <div id="searchSection" class="search-section">
                <label class="search-label" for="voucherInput">
                    📱 Weka Voucher Yako Hapa:
                </label>
                <div class="search-group">
                    <input 
                        type="text" 
                        id="voucherInput" 
                        class="search-input" 
                        placeholder="Andika voucher yako"
                    >
                    <button id="searchBtn" class="search-btn">🔍 Tafuta</button>
                </div>
                <div id="errorMsg" class="error" style="display:none;"></div>
            </div>
            
            <div id="progressSection" class="progress-section">
                <h3>Inaendelea...</h3>
                <div class="progress-steps">
                    <div class="progress-step" id="step1">
                        <div>
                            <span class="step-icon">1️⃣</span>
                            <strong>Kufuta voucher ya zamani</strong>
                        </div>
                        <div id="step1-status"></div>
                    </div>
                    <div class="progress-step" id="step2">
                        <div>
                            <span class="step-icon">2️⃣</span>
                            <strong>Kufuta cookies</strong>
                        </div>
                        <div id="step2-status"></div>
                    </div>
                    <div class="progress-step" id="step3">
                        <div>
                            <span class="step-icon">3️⃣</span>
                            <strong>Kuongeza voucher mpya</strong>
                        </div>
                        <div id="step3-status"></div>
                    </div>
                </div>
                <div id="errorMsgProgress" class="error" style="display:none;"></div>
                <button id="retryBtn" class="retry-btn" style="display:none;">🔄 Jaribu Tena</button>
            </div>
            
            <div id="resultSection" class="result">
                <div class="user-info">
                    <h3>📋 Voucher Imepatikana:</h3>
                    <div id="userDetails" class="user-details"></div>
                </div>
                <button id="repairBtn" class="repair-btn">
                    🔧 Bonyeza Hapa Kurekebisha
                </button>
            </div>
            
            <div id="successSection" class="success">
                <div class="success-icon">✅</div>
                <h2>Hongera!</h2>
                <div class="voucher-display">
                    <div>Voucher Yako:</div>
                    <div id="finalVoucher" class="voucher-code"></div>
                    <div>Password:</div>
                    <div id="finalPassword" class="voucher-code"></div>
                </div>
                <p><strong>Vocha yako ipo Tayari! Asante kwa kutumia. Jiache WiFi.</strong></p>
                <button id="copyRedirectBtn" class="copy-redirect-btn">
                    📋 Bonyeza Hapa Kuicopy na Kwenda Kuiweka ili Utumie
                </button>
            </div>
        </div>
    </div>

    <script>
        let foundUser = null;
        let currentStep = 0;
        let failedStep = null;
        
        // Search button handler
        document.getElementById('searchBtn').addEventListener('click', function() {
            const voucher = document.getElementById('voucherInput').value.trim();
            if (voucher.length >= 7) {
                searchVoucher(voucher);
            } else {
                showError('Weka voucher yako (angalau herufi 7)');
            }
        });
        
        // Enter key on input
        document.getElementById('voucherInput').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                document.getElementById('searchBtn').click();
            }
        });
        
        function searchVoucher(voucher) {
            document.getElementById('errorMsg').style.display = 'none';
            document.getElementById('resultSection').style.display = 'none';
            
            fetch(window.location.href, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=search_voucher&voucher=${encodeURIComponent(voucher)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    foundUser = data.user;
                    showResult(data.user);
                } else {
                    showError(data.message);
                }
            })
            .catch(error => {
                showError('Kuna tatizo la mtandao. Jaribu tena.');
            });
        }
        
        function showResult(user) {
            const userDetails = document.getElementById('userDetails');
            userDetails.innerHTML = `
                <p><strong>Jina:</strong> ${user.name}</p>
                <p><strong>Password:</strong> ${user.password}</p>
                <p><strong>Profile:</strong> ${user.profile}</p>
                <p><strong>Server:</strong> ${user.server}</p>
                ${user.comment ? '<p><strong>Comment:</strong> ' + user.comment + '</p>' : ''}
            `;
            
            document.getElementById('resultSection').style.display = 'block';
        }
        
        function showError(message) {
            const errorMsg = document.getElementById('errorMsg');
            errorMsg.textContent = message;
            errorMsg.style.display = 'block';
        }
        
        // Repair button handler
        document.getElementById('repairBtn').addEventListener('click', function() {
            if (!foundUser) return;
            
            document.getElementById('resultSection').style.display = 'none';
            document.getElementById('searchSection').style.display = 'none';
            document.getElementById('progressSection').style.display = 'block';
            
            currentStep = 0;
            failedStep = null;
            resetProgressSteps();
            startRepairProcess();
        });
        
        function resetProgressSteps() {
            ['step1', 'step2', 'step3'].forEach(id => {
                document.getElementById(id).className = 'progress-step';
                document.getElementById(id + '-status').innerHTML = '';
            });
            document.getElementById('errorMsgProgress').style.display = 'none';
            document.getElementById('retryBtn').style.display = 'none';
        }
        
        function startRepairProcess() {
            executeStep('remove_user', 1);
        }
        
        function executeStep(stepName, stepNum) {
            currentStep = stepNum;
            
            // Mark step as active
            const stepEl = document.getElementById('step' + stepNum);
            stepEl.className = 'progress-step active';
            stepEl.querySelector('[id$="-status"]').innerHTML = '<div class="spinner"></div>';
            
            fetch(window.location.href, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=repair_step&step=${stepName}&user_data=${encodeURIComponent(JSON.stringify(foundUser))}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Mark step as completed
                    stepEl.className = 'progress-step completed';
                    stepEl.querySelector('[id$="-status"]').innerHTML = '✅';
                    
                    // Move to next step
                    if (data.step_completed === 'remove_user') {
                        executeStep('remove_cookies', 2);
                    } else if (data.step_completed === 'remove_cookies') {
                        executeStep('add_user', 3);
                    } else if (data.step_completed === 'add_user') {
                        // All done!
                        showSuccess(data.voucher, data.password);
                    }
                } else {
                    // Mark step as failed
                    failedStep = stepName;
                    stepEl.className = 'progress-step failed';
                    stepEl.querySelector('[id$="-status"]').innerHTML = '❌';
                    
                    const errorMsgProgress = document.getElementById('errorMsgProgress');
                    errorMsgProgress.textContent = data.message;
                    errorMsgProgress.style.display = 'block';
                    document.getElementById('retryBtn').style.display = 'inline-block';
                }
            })
            .catch(error => {
                failedStep = stepName;
                stepEl.className = 'progress-step failed';
                stepEl.querySelector('[id$="-status"]').innerHTML = '❌';
                
                const errorMsgProgress = document.getElementById('errorMsgProgress');
                errorMsgProgress.textContent = 'Kuna tatizo la mtandao. Jaribu tena.';
                errorMsgProgress.style.display = 'block';
                document.getElementById('retryBtn').style.display = 'inline-block';
            });
        }
        
        // Retry button handler
        document.getElementById('retryBtn').addEventListener('click', function() {
            if (failedStep === 'remove_user') {
                resetProgressSteps();
                executeStep('remove_user', 1);
            } else if (failedStep === 'remove_cookies') {
                // Mark step 1 as already completed
                document.getElementById('step1').className = 'progress-step completed';
                document.getElementById('step1-status').innerHTML = '✅';
                resetStepDisplay(2);
                executeStep('remove_cookies', 2);
            } else if (failedStep === 'add_user') {
                // Mark steps 1 and 2 as already completed
                document.getElementById('step1').className = 'progress-step completed';
                document.getElementById('step1-status').innerHTML = '✅';
                document.getElementById('step2').className = 'progress-step completed';
                document.getElementById('step2-status').innerHTML = '✅';
                resetStepDisplay(3);
                executeStep('add_user', 3);
            }
        });
        
        function resetStepDisplay(stepNum) {
            const stepEl = document.getElementById('step' + stepNum);
            stepEl.className = 'progress-step';
            stepEl.querySelector('[id$="-status"]').innerHTML = '';
            document.getElementById('errorMsgProgress').style.display = 'none';
            document.getElementById('retryBtn').style.display = 'none';
        }
        
        function showSuccess(voucher, password) {
            document.getElementById('progressSection').style.display = 'none';
            document.getElementById('finalVoucher').textContent = voucher;
            document.getElementById('finalPassword').textContent = password;
            document.getElementById('successSection').style.display = 'block';
            
            // Store voucher info in localStorage
            localStorage.setItem('lastVoucher', voucher);
            localStorage.setItem('lastPassword', password);
        }
        
        // Copy and redirect button
        document.getElementById('copyRedirectBtn').addEventListener('click', function() {
            const voucher = localStorage.getItem('lastVoucher');
            const password = localStorage.getItem('lastPassword');
            
            if (voucher && password) {
                const textToCopy = `Username: ${voucher}\nPassword: ${password}`;
                
                // Try to copy to clipboard
                if (navigator.clipboard) {
                    navigator.clipboard.writeText(textToCopy).then(() => {
                        alert('Voucher imecopy! Sasa utapelekwa kwenye router.');
                        window.location.href = 'http://10.0.0.1';
                    }).catch(() => {
                        alert(`Username: ${voucher}\nPassword: ${password}\n\nCopy hii info na uweke kwenye router.`);
                        window.location.href = 'http://10.0.0.1';
                    });
                } else {
                    alert(`Username: ${voucher}\nPassword: ${password}\n\nCopy hii info na uweke kwenye router.`);
                    window.location.href = 'http://10.0.0.1';
                }
            }
        });
    </script>
</body>
</html>